#include <Wire.h>                           // ESP8266's I2C library, necessary to setup the Luminosity Sensor
#include <Adafruit_Sensor.h>                // Unified Adafruit's library for sensors, to properly normalise Luminosity Sensor's input
#include <Adafruit_TSL2561_U.h>             // Adafruit's library for handling TSL2561 Luminosity Sensor

#include <ESP8266WiFi.h>                    // ESP8266's WiFi handling library
#include <ESP8266HTTPClient.h>              // ESP8266's HTTP Client, for conection with the Webserver
#include <DNSServer.h>                      // Local DNS Server used for redirecting all requests to the configuration portal
#include <ESP8266WebServer.h>               // Local WebServer used to serve the configuration portal
#include <WiFiManager.h>                    // Opensource library allowing for easier WiFi access point handling and remote access point selection 
#include <ArduinoJson.h>                    // JSON parsing and/or encoding

#include <EEPROM.h>                         // Persistent memory storage
#include <Ticker.h>                         // Provides non-primitive path for conditioned looped execution, e.g. diode blinking during setup. 

const char* SERVER_HOST = "104.197.13.227"; // Webserver IP address
const int   SERVER_PORT = 84;               // Weberver API port
const int   SERVER_CALL_INTERVAL = 20;      // Webserver communication interval (in sec)
const String pingEP = "/iot/ping";          // Webserver endpoint for setup-phase ping
const String feedEP = "/iot/checkin";       // Webserver endpoint for continuious feed

bool SERVER_FEED_TIME = true;               // Control variable used to mark the server communication interval accurence
bool LOW_WATER = false;                     // Control variable used to mark low level of water in the device, for internal purposes

// Webserver entities, necessary to enable the WiFi selection panel

HTTPClient http;                            
ESP8266WebServer server(80);

// Pinouts map definitions

const int WLS_IN = A0;
const int TSL_SDA = D4;
const int TSL_SCL = D5;
const int LED_ONE = D6;
const int LED_TWO = D7;
const int RGB_RED = D3;
const int RGB_BLUE = D2;
const int RGB_GREEN = D1;
const int PUMP_PIN = D8;

const float WLS_MAX = 580.0;                // Maximum average reading obtained from the Water Level Sensor completely submerged
const float WLS_MIN = 120.0;                // Minimum average reading obtained from the Water Level Sensor barely in touch with water
const float WLS_WARNING = 15;               // Percentage treshold of the water level reading at which the device warns the user with the RGB LED

// RGB LED colours definitions

static int OFF[] = {0, 0, 0};
static int WHITE[] = {PWMRANGE, PWMRANGE, PWMRANGE};
static int RED[] = {PWMRANGE, 0, 0};
static int GREEN[] = {0, PWMRANGE, 0};
static int BLUE[] = {0, 0, PWMRANGE};
static int ORANGE[] = {PWMRANGE, PWMRANGE * 60 / 256, PWMRANGE * 30 / 256};
static int CYAN[] = {PWMRANGE * 66/255, PWMRANGE * 244/255, PWMRANGE * 222/255};

// Luminosity Senor mode of operation definition, for increased readability

tsl2561IntegrationTime_t TSL_Fast = TSL2561_INTEGRATIONTIME_13MS;
tsl2561IntegrationTime_t TSL_Mixed = TSL2561_INTEGRATIONTIME_101MS;
tsl2561IntegrationTime_t TSL_Precise = TSL2561_INTEGRATIONTIME_402MS;

// Luminosity Sensor related variables, for internal use

Adafruit_TSL2561_Unified tsl = Adafruit_TSL2561_Unified(TSL2561_ADDR_FLOAT, 12345);
sensors_event_t TSL_event;

// Ticker entities, allowing operations taking place every X seconds

Ticker ticker;
Ticker serverTicker;

// Variables related to the restart fallback mechanism

byte rstCnt = 0;
int rstCntAdd = 0;
const int rstLimit = 3;

//--------------------------------------------------------------------------------------------------------------------------------------------------

void setup() {                              // Setup code here, to run once
  Serial.begin(9600);                       // Open debug feed channel at given frequency
  delay(10); 

  Serial.println(""); 
  Serial.println("------------------------------------");
  Serial.println("------------------------------------");
  Serial.println("VEREATABLE Indoor Garden - BETA");
  Serial.println("------------------------------------");
  Serial.println("------------------------------------");
  Serial.println("");

  //Initialise all components and appropriate segments of code

  initEeprom();
  initTsl();
  initWls();
  initLeds();
  initPump();
  initWifi();
  pingServer();
  initFeed();
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void initEeprom() {                         // Open persistent memory
  EEPROM.begin(512);
}

void initTsl() {                            // Tests Luminosity Sensor's connection and initialises it
  Wire.begin(TSL_SDA, TSL_SCL);

  if (tsl.begin()) {
    tsl.enableAutoRange(true);              // Initialises Luminosity Sensor to automatic range setting
    tsl.setIntegrationTime(TSL_Fast);       // Initialises Luminosity Sensor according to the mode selected
  
    Serial.println("------------------------------------");
    Serial.println("Luminosity Sensor initialised");
    Serial.println("------------------------------------"); Serial.println("");
    rgbBlink(GREEN, 1);                     // Mark operation success to the user and to the fallback mechanism
    clearRestartCounter();
  } else {
    Serial.println("------------------------------------");
    Serial.println("Luminosity Sensor not found ... Check your wiring or I2C ADDR!");
    Serial.println("------------------------------------"); Serial.println("");
    restartDevice();                        // Restart the device through the fallback mechanism
  }
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void initWls() {                            // Initialise the Water Level Sensor
  pinMode(WLS_IN, INPUT);
  Serial.println("------------------------------------");
  Serial.println("Water Level Sensor initialised");
  Serial.println("------------------------------------"); Serial.println("");
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void initLeds() {                           // Initialise the LED components
  pinMode(LED_ONE, OUTPUT);
  pinMode(LED_TWO, OUTPUT);
  pinMode(RGB_RED, OUTPUT);
  pinMode(RGB_BLUE, OUTPUT);
  pinMode(RGB_GREEN, OUTPUT);
  Serial.println("------------------------------------");
  Serial.println("LED Elements initialised");
  Serial.println("------------------------------------"); Serial.println("");
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void initPump() {                           // Initialise the Waterpump 
  pinMode(PUMP_PIN, OUTPUT);
  Serial.println("------------------------------------");
  Serial.println("Waterpump Motor initialised");
  Serial.println("------------------------------------"); Serial.println("");
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void initWifi() {                           // Initialise wireless connectivity
  WiFiManager wifiManager;
//  wifiManager.resetSettings();            // Keep commented unless clearing the settings is required for troubleshooting
  wifiManager.setTimeout(180);
  wifiManager.setDebugOutput(false);
  wifiManager.setAPCallback(configModeCallback);
  
  if (!wifiManager.autoConnect("Vereatable WiFi Setup")) {      // If necessary, will create a wireless access point named "Vereatable WiFi Setup", where new WiFi connection can be selected
    Serial.println("------------------------------------");     // If failed to open the access point or reached 3-minutes timeout of its operation, restart the device using the fallback mechanism
    Serial.println("Wireless Manager event: Failed to connect and hit timeout");
    Serial.println("------------------------------------"); Serial.println("");
    ticker.detach();
    restartDevice();
  }

  Serial.println("------------------------------------");       // Pass the details of a successful connection to debug
  Serial.println("Wireless Manager event: Connected successfully to " + WiFi.SSID() + " at IP " + WiFi.localIP() + " with MAC " + WiFi.macAddress());
  Serial.println("------------------------------------"); Serial.println("");
  ticker.detach();
  rgbBlink(GREEN, 1);                       // Mark operation success to the user and to the fallback mechanism
  clearRestartCounter();
}

void configModeCallback (WiFiManager *myWiFiManager) {
  Serial.println("------------------------------------");
  Serial.println("Wireless Manager event: No known access point available, entering setup: " + String(WiFi.softAPIP()) + " ");
  Serial.println(myWiFiManager->getConfigPortalSSID());
  Serial.println("------------------------------------"); Serial.println("");
  ticker.attach_ms(250, rgbTick, CYAN);     // Indicate the WiFi configuration access point being active
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void initFeed() {
  serverTicker.attach(SERVER_CALL_INTERVAL, feedTick);          // Start the automated scheduler of the device-server communication
}

void feedTick() {
  SERVER_FEED_TIME = true;                                      // Mark the need to communicate with the server 
  Serial.println("------------------------------------");
  Serial.println("Wireless Manager event: Time to send feed to server");
  Serial.println("------------------------------------"); Serial.println("");
}


//--------------------------------------------------------------------------------------------------------------------------------------------------

void pingServer() {

  // Prepare connection with the server in order to check its availability with a simple ping call

  String serverRequest = pingEP;
  http.begin(SERVER_HOST, SERVER_PORT, serverRequest);
  int httpCode = http.GET(); 
  
  Serial.println("------------------------------------");
  Serial.println("Wireless Manager event: Sending GET request to " + String(SERVER_HOST) + " at port " + String(SERVER_PORT));
  Serial.println("------------------------------------"); Serial.println("");
  
  if (httpCode) {
    if (httpCode == 200) {
      String payload = http.getString();                        // Server response OK
      Serial.println("------------------------------------");
      Serial.println("Wireless Manager event: Server response ("+ String(httpCode) + ")");
      Serial.println("------------------------------------"); Serial.println("");
      rgbBlink(GREEN, 1);                                       // Mark operation success to the user and to the fallback mechanism
      clearRestartCounter();
    } else {
      Serial.println("------------------------------------");   // Server not found or responded with an error
      Serial.println("Wireless Manager event: Failed to communicate with the server (" + String(httpCode)  + ")");
      Serial.println("------------------------------------"); Serial.println("");
      restartDevice();                                          // Restart the device through the fallback mechanism
    }
  }
  http.end();
}

void feedServer() {

  // Prepare connection with the server in order to check its availability with a simple ping call

  http.begin(SERVER_HOST, SERVER_PORT, feedEP);
  http.addHeader("Content-Type", "application/json");

  int WLS_reading;
  int TSL_reading;
  String jsonString;

  WLS_reading = getWlsReading();                                // Get a reading from the Water Level Sensor
  TSL_reading = getTslReading();                                // Get a reading from the Luminosity Sesnor

  StaticJsonBuffer<200> jsonBuffer;                             // Parse JSON request object with the sensor readings
  JsonObject& request = jsonBuffer.createObject();
  request["SerialNumber"] = String(ESP.getChipId());;
  request["LightLevel"] = TSL_reading;
  request["WaterLevel"] = WLS_reading;
  request.printTo(jsonString);

  if (LOW_WATER) {                                              // If low water warning is on, supress it until the end of web communiction
    ticker.detach();
  }
  ticker.attach_ms(250, rgbTick, WHITE);                        // Indicate ongoing network operation
  
  Serial.println("------------------------------------");
  Serial.println("Wireless Manager event: Sending POST request to " + String(SERVER_HOST) +" at port " + String(SERVER_PORT));
  Serial.println(jsonString);
  Serial.println("------------------------------------"); Serial.println("");
  int httpCode = http.POST(jsonString);
 
  if (httpCode) {
    if (httpCode == 200) {                                      // Server response OK
      String payload = http.getString();
      Serial.println("------------------------------------");
      Serial.println("Wireless Manager event: Server response (" + String(httpCode) + ")");
      Serial.println(payload);
      Serial.println("------------------------------------"); Serial.println("");
      parseControlObject(payload);                              // Apply control data obtained from the server
      ticker.detach();
      rgbBlink(GREEN, 1);
    } else {
      Serial.println("------------------------------------");  // Server not found or responded with an error
      Serial.println("Wireless Manager event: Failed to communicate with the server (" + String(httpCode) + ")");
      Serial.println("------------------------------------"); Serial.println("");
      ticker.detach();
      rgbBlink(RED, 3);                                        // Indicate error without restart with hope for
    }
  }
  if (LOW_WATER) {
    ticker.attach_ms(500, rgbTick, BLUE);                   // After sucesfull communication, return to the low water warning if needed
  }
  http.end();
}

int parseControlObject(String json) {
  const size_t bufferSize = JSON_OBJECT_SIZE(6) + 250;        // Parse the control response from JSON body
  DynamicJsonBuffer jsonBuffer(bufferSize);
  JsonObject& control = jsonBuffer.parseObject(json);
  
  if(!control.success()) {
    Serial.println("------------------------------------");
    Serial.println("Wireless Manager event: Failed to parse control object response");
    Serial.println("------------------------------------"); Serial.println("");
    return 1;
  }

  int ledOneDuty = control["dutyUpperLed"];
  int ledTwoDuty = control["dutyLowerLed"];         
  double pumpUptime = control["pumpDuration"];

  pwm(LED_ONE, ledOneDuty);                                   // Apply received settings to upper LED bar
  pwm(LED_TWO, ledTwoDuty);                                   // Apply received settings to lower LED bar
  if (pumpUptime > 0) {
    pwm(PUMP_PIN, 100, pumpUptime);                           // Apply received settings to the water pump, if any received
  }
  
  return 0;
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void loop() {                                                 // Main code here, to run repeatedly
  if (SERVER_FEED_TIME) {
    SERVER_FEED_TIME = false;
    feedServer();                                             // Reach out to the server if the time is appropriate
  }
  delay(100);
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void rgb(int colour[3]) {                                     // Control the RGB LED, apply colour given
  analogWrite(RGB_RED, colour[0]);
  analogWrite(RGB_BLUE, colour[1]);
  analogWrite(RGB_GREEN, colour[2]);
}

void rgbBlink(int colour[3], int duration) {                  // Blik the RGB LED with given colour and duration
  rgb(colour);
  delay(duration * 1000);
  rgb(OFF);
}

void rgbTick(int colour[3]) {                                 // Toggle RGB LED between given colour and no light, used for frequency blinking
  if (digitalRead(RGB_RED) || digitalRead(RGB_GREEN) || digitalRead(RGB_BLUE)) {
     rgb(OFF);
  } else {
    rgb(colour);
  }
}

void pwm(int pwmPin, int pwmPercent) {                        // Apply Pulse Width Modulation percentage value to given pin
  int pwmValue;
  if (pwmPercent >= 100) {
    pwmValue = PWMRANGE;
  } else {
    pwmValue = PWMRANGE / 100 * pwmPercent;
  }
  analogWrite(pwmPin, pwmValue);
}

void pwm(int pwmPin, int pwmPercent, double pwmDuration) {    // Apply Pulse Width Modulation percentage value to given pin, for given time
  pwm(pwmPin, pwmPercent);
  delay(pwmDuration * 1000);
  pwm(pwmPin, 0);
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

int getTslReading() {                                         // Obtain reading from the Luminosity Sensor
  tsl.getEvent(&TSL_event);
  Serial.print("Luminosity Sensor event: ");
  if (TSL_event.light) {
    Serial.println(String(TSL_event.light) + " lux");         // Prints light level value in luxs
  } else {
    Serial.println("Sensor saturation");                      // 0-leveled input most likely means saturation
  }
  return TSL_event.light;
}

int getWlsReading() {                                         // Obtain reading from the Water Level Sensor
  int WLS_event;
  float WLS_event_normalised;
  
  WLS_event = analogRead(WLS_IN);                             // Recalculate the raw input into a percentage of submersion, according to the formula found during the testing phase of the project
  WLS_event_normalised = 0.0381 * exp(0.0059 * WLS_event) * 100;
  if (WLS_event_normalised >= 100) { WLS_event_normalised = 100; }
  if (WLS_event_normalised <= 0) { WLS_event_normalised = 0; }
  if (WLS_event_normalised <= WLS_WARNING) {
    LOW_WATER = true;
    ticker.attach_ms(500, rgbTick, BLUE);                  // Activate low water level warning if detected
  } else {
    LOW_WATER = false;
    ticker.detach();
  }
  Serial.println("Waterlevel Sensor event: " + String(WLS_event) + " / "+ String(WLS_event_normalised) + " %");
  return WLS_event_normalised;
}

//--------------------------------------------------------------------------------------------------------------------------------------------------

void restartDevice() {                                        // Restart the device, count the following restarts. Shut down if exceeded limit of restarts
  rstCnt = EEPROM.read(rstCntAdd);
  Serial.println("------------------------------------");
  Serial.println("System event: Approaching post-failure restart. Restart counter: " + String(rstCnt));
  Serial.println("------------------------------------"); Serial.println("");
  if (rstCnt < rstLimit) {
    rstCnt = rstCnt + 1;
    EEPROM.write(rstCntAdd, rstCnt);
    EEPROM.commit();
    EEPROM.end();
    delay(100);
    rgbBlink(RED, 3);
    delay(100);
    ESP.restart();
    delay(100);
  } else {
    rgbBlink(RED, 3);
    ESP.deepSleep(999999999*999999999U, WAKE_NO_RFCAL);
  }
}

void clearRestartCounter() {                                    // Clear the restart counter after sucessfully passing a initialisation step
  Serial.println("------------------------------------");
  Serial.println("System event: Clearing the restart counter");
  Serial.println("------------------------------------"); Serial.println("");
  EEPROM.write(rstCntAdd, 0);
  EEPROM.commit();
}

//--------------------------------------------------------------------------------------------------------------------------------------------------


